.module      UHF3_UTL;
.PAGELENGTH  59;                    { Works for .LST files with 550C }
.include    <system.k>;

{ Conditional assembly for putting code into EPROM, along with monitor.
  If EPROM, no .exe file is used. So, uhf3.exe is the non-EPROM version.
  1=Being put into EPROM, 0=Being loaded from PC }
#define  EPROM  0

{ One and only one CODEC should be used: }
#define  CODEC_1847 1
#define  CODEC_1885 0
#define  CODEC_AIC23 0

#define EZKIT 1
#define DSPX 0

{************************************************************************
 *
 *        EZ-KIT Lite DSP Program for DSP-10 2-meter Transciever
 *			UHF3_UTL.DSP Version 3.2
 *		 Utility and Communications Routines
 *
 * NOTICE: Copyright (C) 1996-2005  by Robert S. Larkin
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, version 2 of the License.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * The program author can be contacted by email at boblark@proaxis.com.
 * Alternatively contact the author at
 *     Robert S. (Bob) Larkin
 *     2982 N. W. Acacia Place
 *     Corvallis, OR  97330  USA
 *
 * For more details on the background or use of the program, pay a visit to
 * http://www.proaxis.com/~boblark/dsp10.htm
 *
 * Some parts are based on routines from Analog Devices Applications
 * Using the ADSP 2100 Family, Vol 1 and 2.
 *
 * Serial Communications is structured quite differently from EZ-KIT Lite
 * routines.  Here  they are based on the regular service at 9600 Hz
 * rather than using the timer.  This guarantees the regular servicing
 * of interrupt routines as they all sync to the 48 KHz Codec interrupt.
 *
 *  30 Oct 96   Bob Larkin W7PUA - Separated from UHF3 module
 *  11 Jan 97   Compatible with V1.0 of UHF3.DSP - RSL
 *  12 Sept99   Made versions consistent RSL
 *  16 Oct 00   Added tukey win, triggered meas, stuff for variable
 *              sample rate FFT. RSL
 *  3  May 03   Incorporated Lyle Johnson DSPx CODEC stuff. RSL
 *  22 Jan 04   Revised wfir1 for audio transmit filter  RSL
 *  21 Nov 04   Changed last status to 0A (rev 2A) RSL
 *  24 Sept05   Added PF1 for EZKIT only, see UHF3A.DSP RSL
 ************************************************************************}

.const      TAPS1=130;                  { FIR1 - 2.8 KHz LPF, 48kHz rate}
.const      TAPS2=130;                  { FIR2 - 2.8 Interpolation LPF }
.const	    TAPS3=199;			{ FIR3 - Audio fir, 9.6 kHz rate }
.const      TAPS4=130;                  { FIR4 - BPF for transmit audio }
.const      TAPS5=200;                  { FIR2 - .5KHz LPF, CW Xmit shaping }
.const	    TAPS6=37;			{ FIR6 - 10KHz HP filt for FM }
.const	    TAPS7=30;			{ FIR7 - 6 KHz LP for FM }
.const	    TAPS8=60;			{ 1200 & 2400 LPF for Nyquist - FFT }
.const      H1=71;			{ Num taps in Hilbert #1 FIR filt }
.const      H1P1ON2=36;			{ This is  (H1+1)/2  }
.const      H1M1ON2=35;			{ ...and (H1-1)/2    }
.const      H1M3ON2=34;			{ ...and (H1-3)/2    }

.const      H3=247;			{ Num taps in Hilbert #3 FIR filt }
.const      H3P1ON2=124;		{ This is  (H3+1)/2  }
.const      H3M1ON2=123;		{ ...and (H3-1)/2    }
.const      H3M3ON2=122;		{ ...and (H3-3)/2    }

.const      N=1024;                     { Number of samples to be FFT'd }
.const	    N_DIV_2=512;
.const      DATA_BLOCKS=3;		{ UART transmit blocks }
.const      ESC_CH=0x24;		{ '$' }

{----------------------------------------------------------------------------}
{          TI  TLV320AIC23B  CODEC Initialization Constants		     }
{----------------------------------------------------------------------------}

.const	C_LT_IN_VOL =	b#0000000100010111;  { Left Input Volume Control
			  |-----|||xx|---|
			  |     |||  |   +---Input Gain LSB 11111=+12 dB, 1.5 dB steps
			  |     |||  +-------Input Gain MSB 00000=-34.5 dB (0 dB shown)
			  |     ||+----------Input Mute (1=mute)
			  |     |+-----------LR Simultaneous Gain/Mute Update (1=enable)
			  +-----+------------Left Input Volume Control Register }

.const	C_RT_IN_VOL =	b#0000001100010111;  { Right Input Volume Control
			  |-----|||xx|---|
			  |     |||  |   +---Input Gain LSB 11111=+12 dB, 1.5 dB steps
			  |     |||  +-------Input Gain MSB 00000=-34.5 dB (0 dB shown)
			  |     ||+----------Input Mute (1=mute)
			  |     |+-----------RL Simultaneous Gain/Mute Update (1=enable)
			  +-----+------------Right Input Volume Control Register }

.const	C_LT_PHONE_VOL =b#0000010101111001;  { Left Headphone Volume Control
			  |-----||||-----|
			  |     ||||     +---Gain LSB 1111111=+6 dB, 1 dB steps
			  |     |||+---------Gain MSB 0110000=-73 dB (mute) (0 dB shown)
			  |     ||+----------Zero Cross Detect (1=mute)
			  |     |+-----------LR Simultaneous Gain/Mute Update (1=enable)
			  +-----+------------Left Headphone Volume Control Register }

.const	C_RT_PHONE_VOL =b#0000011101111001;  { Right Headphone Volume Control
			  |-----||||-----|
			  |     ||||     +---Gain LSB 1111111=+6 dB, 1 dB steps
			  |     |||+---------Gain MSB 0110000=-73 dB (mute) (0 dB shown)
			  |     ||+----------Zero Cross Detect (1=mute)
			  |     |+-----------RL Simultaneous Gain/Mute Update (1=enable)
			  +-----+------------Right Headphone Volume Control Register }

.const	C_AUD_PATH =	b#0000100000010010;  { Audio Path Control
			  |-----||||||||||
			  |     |||||||||+---MIC Boost (0=0dB, 1=20 dB)
			  |     ||||||||+----MIC Mute (1=muted) 
			  |     |||||||+-----ADC Input Select (0=Line, 1=Mic)
			  |     ||||||+------Bypass (1=enabled)
			  |     |||||+-------DAC Select (1=selected, 0= off)
			  |     ||||+--------SideTone Enable (0 = disabled)
			  |     |||+---------STA0 1= - 9 or -18 dB, 0= -6 or -12 dB
			  |     ||+----------STA1 1= -12 or -18 dB, 0= -6 or -9 dB
			  |     |+-----------STA2 1=   0 dB, 0= -6, -9, -12 or -18 dB
			  +-----+------------Audio Path Control Register }

.const	C_DIG_AUD_PATH =b#0000101000000000;  { Digital Audio Path Control
			  |-----|xxxxx||||
			  |     |     |||+---ADC High Pass Filter (1=enabled)
			  |     |     ||+----DEEMP0 (1= 32 or 48 kHz)
			  |     |     |+-----DEEMP1 (1= 44.1 or 48 kHz)
			  |     |     +------DAC Soft Mute (1=enabled)
			  +-----+------------Digital Audio Path Control Register }

.const	C_POWER =	b#0000110000000000;  { Power Down Control
			  |-----|x||||||||
			  |     | |||||||+---Line Input   (1=OFF)
			  |     | ||||||+----Mic Input    (1=OFF)
			  |     | |||||+-----ADC          (1=OFF)
			  |     | ||||+------DAC          (1=OFF)
			  |     | |||+-------Outputs      (1=OFF)
			  |     | ||+--------Oscillator   (1=OFF)
			  |     | |+---------Clock        (1=OFF)
			  |     | +----------Device Power (1=OFF)
			  +-----+------------Power Down Control Register }

{ Rev 30 Reversed L-R DAC lines. C_DIG_FORMAT was:
.const	C_DIG_FORMAT =	b#0000111001000001; }
.const	C_DIG_FORMAT =	b#0000111001100001;  { Digital Audio Interface Format
			  |-----|xx|||||||
			  |     |  ||||||+---FOR0 Data Format
			  |     |  |||||+----FOR1 (11 = DSP, 10= I2S, 01 MSB left aligned, 00 MSB right)
			  |     |  ||||+-----IWL0 Input Word Length
			  |     |  |||+------IWL1 (11=32 bit, 10=24 bit, 01 = 20 bit, 00 = 16 bit)
			  |     |  ||+-------DAC LR Phase (1= Right Channel LRCIN low, 0= high)
			  |     |  ||         in DSP Mode (1= MSB on 2nd Bclk / after LRCIN /, 0=1st Bclk / )
			  |     |  |+--------DAC LR Swap (1=swap, 0=normal)
			  |     |  +---------MS Mode (1=Master, 0=Slave)
			  +-----+------------Digital Audio Interface Format Register }


{.const	C_SAMPLE_RATE =	b#0001000001011110; } {Orig Sample Rate Control, 48 kHz 18.432 crystal every fourth by 3 }
.const	C_SAMPLE_RATE =	b#0001000000000010;   {Alt Sample Rate Control
			  |-----|x||||||||
			  |     | |||||||+---Clock Mode (1=USB, 0=Normal)
			  |     | ||||||+----Base Oversampling Rate (0=256, 1=384 in normal mode)
			  |     | |||||+-----SR0 Sample Rate, see data sheet
			  |     | ||||+------SR1    shown is 48 kHz sample rate with
			  |     | |||+-------SR2    18.432 MHz Xtal and 24 channels/frame
			  |     | ||+--------SR3    for use with the adi dsp chip and cpld
			  |     | |+---------CLKIN  (1=MCLK/2, 0=MCLK)
			  |     | +----------CLKOUT (1=MCLK/2, 0=MCLK)
			  +-----+------------Sample Rate Control Register }

.const	C_DIG_ACT =	b#0001001000000001;  { Digital Interface Activation
			  |-----|xxxxxxxx|
			  |     |        |+--Activate Interface (1=active, 0=inactive)
			  +-----+------------Digital Interface Activation Register }

.const	C_RESET =	b#0001111000000000;  { Reset
			  |-----||-------|
			  |     |+-------|+--Must be all 0's for RESET
			  +-----+------------Reset Register }

.const  SPICLK=	0x0002;	{ PF data bits for SPI port to init CODEC }
.const  SPICS=	0x0004;
.const  SPIDAT=	0x0001;
.const	CODEC=	0x0000;	{ Dummy address, any will do in I/O space }

{----------------------------------------------------------------------------}
{                 end of CODEC Initialization Constants			     }
{----------------------------------------------------------------------------}

.entry	init_util;
.entry	out_block_byte;
.entry	uart_rcvr;
.entry	window;
.entry	sin;
.entry	arctan;
.entry	do_hilbert1;
.entry	do_hilbert2;
.entry	do_hilbert3;
.entry	do_hilbert4;
.entry	do_fir1i;
.entry	do_fir1q;
.entry	interp_filter;
.entry	audio_fir;
.entry	fir_xmt_audio;
.entry	fir_xmt_cw;
.entry	fir10hp;
.entry	fir6lp1, fir6lp2;
.entry	fir1200, fir2400;

#if CODEC_1847
.entry	next_cmd;
#endif

.entry	wfir1i, wfir1q;
.entry	wfir2i, wfir2q;
.entry  getrnd;

.external	out_log;
.external	inputreal0;
.external	inputreal1;
.external	inputreal2;
.external	fft_fcn;

{*****************  PROGRAM MEMORY  *****************}
.var/pm/circ    ham_win[N];
.init           ham_win: <hmhex.102>;
.var/pm/circ	bh4_win[N];		{ Blackman harris 4 term -92dB window}
.init		bh4_win: <bh4hex.102>;
.var/pm/circ	tukey_win[N];		{ Tukey 25 percent window }
.init		tukey_win: <wintuk25.102>;

.var/pm/circ    fir1_coefs[TAPS1];      { 2.8 kHz LPF }
.init           fir1_coefs: <lp2_8.dat>;
.var/pm/circ    fir1a_coefs[TAPS1];      { 4 kHz LPF, less rejection }
.init           fir1a_coefs: <lp4000a.dat>;

.var/pm/circ    wfir2_coefs[130];      { 5 kHz LPF }
.init           wfir2_coefs: <lp5_48.hex>;

.var/pm/circ    fir2_coefs[TAPS2];      { Interp filter }
.init           fir2_coefs: <lp3_7a.dat>;

#if 0
.init		fir3_coefs4: <cw200_8b.dat>;  { 200 Hz BP at 800 Hz }
#endif

.var/pm/circ	fir3_coefs0[TAPS3];	{ Audio filter 0 }
.init		fir3_coefs0: <match600.dat>;  { 600 Hz matched filter W8MQW }
.var/pm/circ	fir3_coefs1[TAPS3];	{ Audio filter 1 }
.init		fir3_coefs1: <cw200_6b.dat>;  { 200 Hz BP at 600 Hz }
.var/pm/circ	fir3_coefs2[TAPS3];	{ Audio filter 2 }
.init		fir3_coefs2: <cw300_6b.dat>;  { 300 Hz BP at 600 Hz }
.var/pm/circ	fir3_coefs3[TAPS3];	{ Audio filter 3 }
.init		fir3_coefs3: <cw450_6b.dat>;  { 450 Hz BP at 625 Hz }
.var/pm/circ	fir3_coefs4[TAPS3];	{ Audio filter 4 }
.init		fir3_coefs4: <cw600_7b.dat>;  { 600 Hz BP at 700 Hz }
.var/pm/circ	fir3_coefs5[TAPS3];	{ Audio filter 5 }
.init		fir3_coefs5: <ssbn1b.dat>;  { 300-2400 Hz BP }
.var/pm/circ	fir3_coefs6[TAPS3];	{ Audio filter 6 }
.init		fir3_coefs6: <ssbm1b.dat>;  { 250-2800 Hz BP }
.var/pm/circ	fir3_coefs7[TAPS3];	{ Audio filter 7 }
.init		fir3_coefs7: <cw300_8b.dat>;  { 300 Hz BP at 800 Hz }

{ An index table to find the coefficients }
.var/pm		fir3ptr[8];
.init   	fir3ptr:^fir3_coefs0, ^fir3_coefs1,
			^fir3_coefs2, ^fir3_coefs3,
			^fir3_coefs4, ^fir3_coefs5,
			^fir3_coefs6, ^fir3_coefs7;

.var/pm/circ    fir4_coefs[TAPS4];      { Audio transmit filter }
.init           fir4_coefs: <lp2_8.dat>;
.var/pm/circ    fir5_coefs[TAPS5];      { CW transmit filter }
.init           fir5_coefs: <lp_5_48.dat>;
.var/pm/circ	fir6_coefs[TAPS6];	{ 10KHz HP filter for FM rcvr }
.init		fir6_coefs: <hp10fir.dat>;
.var/pm/circ	fir7_coefs[TAPS7];	{ 6KHz LPF for FM rcvr }
.init		fir7_coefs: <lp6fir.dat>;
{ The FM xmit filter includes preemphasis, 4 dB at 2600 Hz, and a cutoff
  at 3200 Hz. It is 130 taps long.  If eventually one desires to put
  a 129 tap filter here, the last entry can  be zero and it will work fine.
  The gain of fmxmit8.dat is 4.  }
.var/pm/circ    fir8_coefs[130];      { FM audio transmit filter }
.init           fir8_coefs: <fmxmit8.dat>;

.var/pm/circ	fir1200_coefs[TAPS8];	{ 1.2KHz LPF for Nyquist - 4800 FFT }
.init		fir1200_coefs: <lp1200a.dat>;
.var/pm/circ	fir2400_coefs[TAPS8];	{ 2.4KHz LPF for Nyquist - 2400 FFT }
.init		fir2400_coefs: <lp2400a.dat>;

.var/pm/circ	hilbert1_coeff[H1P1ON2];
.init		hilbert1_coeff: <hil2_22.dat>;
.var/pm/circ	hilbert3_coeff[H3P1ON2];
.init		hilbert3_coeff: <hil_3_48.dat>;
.var/pm         sin_coeff[5];
.init           sin_coeff: H#324000, H#005300, H#AACC00, H#08B700, H#1CCE00;
.var/pm		atn_coeff[5];
.init		atn_coeff: H#28BD00, H#006D00, H#EF3E00, H#08C600, H#FED400;

{***********************************************************************
 * Pointer table - makes important tables & variables available to PC
 * The last entry is a validity check when loading the table to the PC.
 ***********************************************************************}
.var/pm		ptr_table[38];
.global		ptr_table;
.init		ptr_table: ^ham_win, ^bh4_win, ^tukey_win, ^fir1_coefs,
		  ^fir1a_coefs, ^wfir2_coefs, ^fir2_coefs,
		  ^fir3_coefs0, ^fir3_coefs1, ^fir3_coefs2, ^fir3_coefs3,
		  ^fir3_coefs4, ^fir3_coefs5, ^fir3_coefs6, ^fir3_coefs7,
		  ^fir4_coefs, ^fir5_coefs, ^fir6_coefs,
		  ^fir7_coefs, ^fir8_coefs, 0, 0, 
		  ^fir1200_coefs, ^fir2400_coefs,
		  ^hilbert1_coeff, ^hilbert3_coeff,
		  0, 0, 0, 0,
		  ^xmit_status_buf, ^cmd_buf, ^esc_a, 0,
		  ^buf_len_table, ^configuration, ^filter_id, 0X123456;

{***********************************************************************
 * Filter I.D. - makes filter identity available to PC and filter changers.
 * The same identifier is put into the filter characteristic file.
 * 
 * Word 0: Unique identifier number, 16 bit, chosen pseudo randomly, perhaps
 *		using the date/time of design.
 *
 * Filters: fir3_coefs0    Receive filter
 *          fir3_coefs1    Receive filter
 *          fir3_coefs2    Receive filter
 *          fir3_coefs3    Receive filter
 *          fir3_coefs4    Receive filter
 *          fir3_coefs5    Receive filter
 *          fir3_coefs6    Receive filter
 *          fir3_coefs7    Receive filter
 *          fir4_coefs     Transmit SSB
 *          fir8_coefs     Transmit FM
 *	    spare x4
 *	    end identifier 0X543210
 *          
 ***********************************************************************}
.var/pm		filter_id[15];
.global		filter_id;
.init		filter_id:	0X100000, 0X100100, 0X100200, 0X100300,
				0X100400, 0X100500, 0X100600, 0X100700,
				0X100800, 0X100900, 0X000000, 0X000000,
				0X000000, 0X000000, 0X543210;

{ configuration - allows the PC to know what hardware is being used.
  CODEC Bits 2,1,0   000=1847  001=AIC23
  Board Bits 5,4,3   000=EZKIt 001=DSPx
  EPROM Bit 6        0=No EPROM  1=EPROM load
}
.var/pm		configuration;
.global		configuration;
#if (CODEC_1847 & ~EPROM)
.init		configuration:	0X00;
#endif
#if (CODEC_1847 & EPROM)
.init		configuration:  0X40;
#endif
#if CODEC_AIC23
.init		configuration:  0X01;
#endif

{********************   DATA MEMORY     *******************************}
#if CODEC_1847
.var/dm/circ	tx_buf[3];		{ SPORT0: Cmd + L data + R data }
.var/dm/circ 	rx_buf[3];		{ SPORT0: Status + L data + R data }
.global		tx_buf, rx_buf;
.var/dm/circ    init_cmds[13];
#endif
#if CODEC_AIC23
.var/dm/circ	tx_buf[2];		{ SPORT0: L data + R data }
.var/dm/circ 	rx_buf[2];		{ SPORT0: R data + L data *** note reversal! }
.global		tx_buf, rx_buf;
#endif

.var/dm/circ	idata[TAPS1];
.var/dm/circ	widata[TAPS1];   {<<<<<<<<<<<<<<< }
.var/dm/circ	qdata[TAPS1];
.global		idata, qdata;
.var/dm/circ	audio_data[TAPS2];
.global		audio_data;
.var/dm/circ	audio_fir_data[TAPS3];
.global		audio_fir_data;
.var/dm/circ	xmt_audio_data[TAPS4];
.var/dm/circ	xmt_cw_data[TAPS5];
.global		xmt_audio_data, xmt_cw_data;
.var/dm/circ	rcv_fm[TAPS6];
.var/dm/circ	rcv1_fm[TAPS7];
.var/dm/circ	rcv2_fm[TAPS7];
.var/dm/circ	lp1200[TAPS8];
.var/dm/circ	lp2400[TAPS8];
.var/dm/circ	h1delay[H1M1ON2];
.var/dm/circ	h1data[H1];
.global		h1delay, h1data;
.var/dm/circ	h2delay[H1M1ON2];
.var/dm/circ	h2data[H1];
.global		h2delay, h2data;
.var/dm/circ	h3delay[H3M1ON2];
.var/dm/circ	h3data[H3];
.global		h3delay, h3data;
{ See block5() for data structure for sending tx data }
buf_len_table:
.var/dm		tx_buf_len[3];		{ Characters remaining in each buf }
.var/dm		tx_buf_ptr[3];		{ Address of each buffer }
.var/dm		tx_buf_word_len[3];	{ 0 for byte, 1 for word }
.global		tx_buf_len, tx_buf_ptr, tx_buf_word_len;
.var/dm		esc_a[2];		{ 0x31, 'A' Data Prefix }
.global		esc_a;
.var/dm		status_buf[20];
.global		status_buf;
.var/dm		xmit_status_buf[10];	{ Instead of UART_tx_data_buf for xmit}
.global		xmit_status_buf;
.var/dm		cmd_buf[5];		{ No esc_ch, just cmd byte + 4 data }
.global		cmd_buf;		{   for commands from pc to dsp }
.var/dm         fft_buff_count;
.global		fft_buff_count;
.var/dm		do_window;		{ 0 no window, 1=Hamming etc }
.global		do_window;
.var/dm		iq_lpf;			{ 0=2.8 kHz BW, 1=4 kHz BW }
.global		iq_lpf;
.var/dm         integrate_count;	{ Number of non-coherent samples }
.global         integrate_count;
.var/dm		mode_select;		{ 0=ssb, 1=cw, etc }
.global		mode_select;
.var/dm		ssb_select;		{ 0=lsb, 1=usb }
.global		ssb_select;
.var/dm         i0_sav, m0_sav, l0_sav; { Used by interrupting routine }
.var/dm		m1_sav;
.var/dm         i4_sav, m4_sav, l4_sav; {   to save fft dedicated DAG regs }
.global         i0_sav, m0_sav, l0_sav;
.global		m1_sav;
.global         i4_sav, m4_sav, l4_sav;
.var/dm         fir1i_i0_sav;		{ Save DAG0 reg FIR1 }
.var/dm         wfir1i_i0_sav;		{ Save DAG0 reg FIR1 <<<<<<<<<<<<<<<<}
.var/dm         fir1q_i0_sav;
.var/dm         fir2_i0_sav;		{ Save DAG0 reg FIR2 }
.var/dm         fir3_i0_sav;		{ Save DAG0 reg FIR3 }
.global		fir1i_i0_sav, fir1q_i0_sav, fir2_i0_sav, fir3_i0_sav;
.var/dm         fir4_i0_sav;		{ Save DAG0 reg FIR4 }
.var/dm         fir5_i0_sav;		{ Save DAG0 reg FIR5 }
.var/dm         fir6_i0_sav;		{ Save DAG0 reg FIR6 }
.var/dm         fir7_1_i0_sav;		{ Save DAG0 reg FIR7_1 }
.var/dm         fir7_2_i0_sav;		{ Save DAG0 reg FIR7_2 }
.var/dm         fir1200_i0_sav;		{ Save DAG0 reg FIR1200 }
.var/dm         fir2400_i0_sav;		{ Save DAG0 reg FIR2400 }
.global		fir4_i0_sav, fir5_i0_sav;
.var/dm         win_i4_sav, win_m4_sav, win_l4_sav; { Save DAG4 regs window }
.global         win_i4_sav, win_m4_sav, win_l4_sav;
.var/dm         if_dphase;
.var/dm		if_phase;
.global		if_dphase, if_phase;
.var/dm		isave, qsave;
.var/dm		audsav;			{ Save between 9.6 KHz blocks }
.global		isave, qsave, audsav;
.var/dm		audio_fir_on;		{ 1=audio_fir() being called }
.var/dm		audio_fir_select;	{ (0 to 7) coefs }
.var/dm		audio_fir_gain;		{ (0 to 15) }
.global		audio_fir_on, audio_fir_select, audio_fir_gain;
.var/dm		h1delay_i0_sav;
.var/dm		h1data_i0_sav;
.var/dm		h2delay_i0_sav;
.var/dm		h2data_i0_sav;
.global		h1delay_i0_sav, h1data_i0_sav, h2delay_i0_sav, h2data_i0_sav;
.var/dm		h3delay_i0_sav;
.var/dm		h3data_i0_sav;
.global		h3delay_i0_sav, h3data_i0_sav;
.var/dm		block_count;		{ 0 to 4 for 9.6kHz block calls }
.global		block_count;
.var/dm		divide_count;		{ 0 to 1, or, 0 to 3 FFT sample rate }
.global		divide_count;
.var/dm		UART_tx_data_block;	{ Sequential data blocks to send }
.var/dm		UART_tx_ptr;		{ Points to current char being sent }
.var/dm		UART_tx_ch_count;	{ Number of char left; 0=empty }
.global		UART_tx_data_block, UART_tx_ptr, UART_tx_ch_count;
.var/dm		UART_tx_cur_bit;	{ Bit being sent, (0,9) }
.var/dm		UART_tx_cur_data;	{ Current data byte or remaining part }
.var/dm		UART_tx_word_len;	{ 0 for byte, 1 for word }
.global		UART_tx_cur_bit, UART_tx_cur_data, UART_tx_word_len;
.var/dm		UART_tx_byte_num;	{ For words, byte 0 (lsb) or 1 (msb) }
.var/dm		UART_tx_esc;		{ 1 if sending esc char }
.var/dm		UART_tx_resync_count;	{ Skip a ch every 1000 }
.global		UART_tx_byte_num, UART_tx_esc, UART_tx_resync_count;
.var/dm		UART_tx_data_done;	{ 1 in block5 when data finished }
.var/dm		UART_tx_buf_select;	{ 0 or 1 for two tx_data_buf0 or buf1 }
.global		UART_tx_data_done, UART_tx_buf_select;
.var/dm		UART_tx_data_buf0[N_div_2 + 1];	{ Data to be sent to pc }
.var/dm		UART_tx_data_buf1[N_div_2 + 1];	{    plus check sum }
.global		UART_tx_data_buf0, UART_tx_data_buf1;
.var/dm		UART_rx_ctr;		{ Used to space bits 5 samples apart }
.var/dm		UART_rx_cur_bit;	{ Bit 10 to 1 plus 0 for idle }
.var/dm		UART_rx_cur_byte;	{ A place to assemble the rcvd byte }
.global		UART_rx_ctr, UART_rx_cur_bit, UART_rx_cur_byte;
.var/dm		UART_rx_cmd_count;
.var/dm		UART_rx_cmd_buf_ptr;
.global		UART_rx_cmd_count, UART_rx_cmd_buf_ptr;
.var/dm		serial_timeout;
.var/dm		serial_error;
.global		serial_timeout, serial_error;
.var/dm		stat_flag;		{ CODEC Init status }
.global		stat_flag;
.var/dm		inptr0;			{ ptr to one overlapping window }
.var/dm		inptr1;			{  and the other, both being filled }
.var/dm		dataptr;		{ ptr to fft input, full }
.global		inptr0, inptr1, dataptr;
.var/dm		get_data_block;		{ 1=still getting 512 pts for fft }
.global		get_data_block;
.var/dm		seed_msw;		{ For RN gen---rev2.1 }
.var/dm		seed_lsw;

{**************************************************************************
 *			init_util
 * Start up initialization of hardware and software.
 **************************************************************************}
init_util:
{ i7, l7, i6, l6 devoted to auto-buffering in SPORT0.  Initialized
  here. These need protection. }
        i7 = ^rx_buf;
        l7 = %rx_buf;
        m7 = 1;
        i6 = ^tx_buf;
        l6 = %tx_buf;
        m6 = 1;

#if CODEC_1847
{ i5, l5, m5 used in codecInit to load AD1847 registers and then made free. }
        i5 = ^init_cmds;
        l5 = %init_cmds;
        m5 = 1;
#endif

	call init_2181;			{ Init dm for main  program }

        i4=^ham_win;			{ DAG4 for windowing table }
        dm(win_i4_sav)=i4;

{**************************************************************************}
#if CODEC_1847
        ax0=0xC000;  dm(tx_buf)=ax0;    { Initially set MCE }
        ax0=0x0000;  dm(tx_buf+1)=ax0;
        ax0=0x0000;  dm(tx_buf+2)=ax0;
#endif
#if CODEC_AIC23
	ax0=0x0000;  dm(tx_buf)=ax0;
	ax0=0x0000;  dm(tx_buf+1)=ax0;
#endif

#if CODEC_1847
{  AD1847  Command Register Initialization }
        ax0=0xc002;  dm(init_cmds)=ax0;
        ax0=0xc102;  dm(init_cmds+1)=ax0;
        ax0=0xc288;  dm(init_cmds+2)=ax0;
        ax0=0xc388;  dm(init_cmds+3)=ax0;
        ax0=0xc488;  dm(init_cmds+4)=ax0;
        ax0=0xc588;  dm(init_cmds+5)=ax0;
        ax0=0xc680;  dm(init_cmds+6)=ax0;
        ax0=0xc780;  dm(init_cmds+7)=ax0;
        ax0=0xc85c;  dm(init_cmds+8)=ax0;  {48 KHz, c850h for 8 KHz }
        ax0=0xc909;  dm(init_cmds+9)=ax0;
        ax0=0xca00;  dm(init_cmds+10)=ax0;
        ax0=0xcc40;  dm(init_cmds+11)=ax0;
        ax0=0xcd00;  dm(init_cmds+12)=ax0;
#endif

{================== S E R I A L   P O R T   #0   S T U F F ==================}
        ax0= b#0000110101111111;    dm (SPORT0_Autobuf) = ax0;
        ax0 = 0;		    dm (SPORT0_RFSDIV) = ax0;
        ax0 = 0;		    dm (SPORT0_SCLKDIV) = ax0;

#if CODEC_1847
        ax0 = b#1000011000001111;   dm (SPORT0_Control_Reg) = ax0;
        ax0 = b#0000000000000111;   dm (SPORT0_TX_Channels0) = ax0;
        ax0 = b#0000000000000111;   dm (SPORT0_TX_Channels1) = ax0;
        ax0 = b#0000000000000111;   dm (SPORT0_RX_Channels0) = ax0;
        ax0 = b#0000000000000111;   dm (SPORT0_RX_Channels1) = ax0;
#endif

#if CODEC_AIC23
{*} {   ax0 = b#1000001000001111;   dm (SPORT0_Control_Reg) = ax0; }{32 ch}

{ Bit15=1 sets multichannel mode for SPORT 0. See User's Manual }
{*}     ax0 = b#1000000000001111;   dm (SPORT0_Control_Reg) = ax0; {24ch}

 { Orig ax0 = b#0001000100010001;   dm (SPORT0_TX_Channels0) = ax0;} { every fourth 48/96 ok }
 {  "   ax0 = b#0000000000010001;   dm (SPORT0_TX_Channels1) = ax0;} { every fourth 48/96 ok }
 {  "   ax0 = b#0001000100010001;   dm (SPORT0_RX_Channels0) = ax0;} { every fourth 48/96 ok }
 {  "   ax0 = b#0000000000010001;   dm (SPORT0_RX_Channels1) = ax0;} { every fourth 48/96 ok }

{The following 4 words map channels to auto-buffering functions. }
{*}     ax0 = b#0101010101010101;   dm (SPORT0_TX_Channels0) = ax0; 
{*}     ax0 = b#0101010101010101;   dm (SPORT0_TX_Channels1) = ax0; 
{*}     ax0 = b#0101010101010101;   dm (SPORT0_RX_Channels0) = ax0; 
{*}     ax0 = b#0101010101010101;   dm (SPORT0_RX_Channels1) = ax0; 
#endif

{============== S Y S T E M   A N D   M E M O R Y   S T U F F ==============}

#if CODEC_1847
	ax0 = 0;
#endif
#if CODEC_AIC23
	ax0 = 0x0FFF;
#endif

	dm(DM_Wait_Reg)=ax0;

        ax0 = b#0001000000000000;	{ Enable SPORT0 }
        dm(System_Control_Reg) = ax0;
        ifc = b#0000000011111111;	{ Clear any pending interrupt }
        nop;
	icntl = 0;
	mstat = b#1000000;
#if EZKIT
	ax0 = b#0111111111000010;	{ Make PF0,2,3,4,5 in  PF1,6,7 out }
#else
	ax0 = b#0111111111000000;	{ Make PF0-PF5=in  PF6 & PF7=out }
#endif
	dm(PFTYPE) = ax0;
	reset FL1;

#if CODEC_AIC23
{----------------------------------------------------------------------------}
{                  TLV320AIC23 Codec intialization			     }
{----------------------------------------------------------------------------}
 	ax0=C_RESET;		call CODEC_CMD;	{ Reset }
	ax0=C_LT_IN_VOL;	call CODEC_CMD;	{ Left Input Volume Control }
	ax0=C_RT_IN_VOL;	call CODEC_CMD;	{ Right Input Volume Control }
	ax0=C_LT_PHONE_VOL;	call CODEC_CMD;	{ Left Headphone Volume Control }
	ax0=C_RT_PHONE_VOL;	call CODEC_CMD;	{ Right Headphone Volume Control }
	ax0=C_AUD_PATH;		call CODEC_CMD;	{ Audio Path Control }
	ax0=C_DIG_AUD_PATH;	call CODEC_CMD;	{ Digital Audio Path Control }
	ax0=C_POWER;		call CODEC_CMD;	{ Power Down Control }
	ax0=C_DIG_FORMAT;	call CODEC_CMD;	{ Digital Audio Interface Format }
	ax0=C_SAMPLE_RATE;	call CODEC_CMD;	{ Sample Rate Control }
	ax0=C_DIG_ACT;		call CODEC_CMD;	{ Digital Interface Activation }

{*}{Orig ax0=SPICS+SPIDAT;	io(CODEC)=ax0;}	{ set CPLD for 48/96 kHz sampling ok }
{*}	ax0=SPICS+SPIDAT+SPICLK;
	io(CODEC)=ax0;			{ set CPLD for 24/48 kHz sampling ok }

 {Question for Lyle - should the next instr have 2 more 1's??????????  } 
        ifc = b#000000111111;     { clear any pending interrupt }
        nop;
 {Q for Bob - IRQE should be set by EPROM? }
        imask = b#0000100000;       { enable rx0 interrupt }
            {     |||||||||+ | timer
                  ||||||||+- | SPORT1 rec or IRQ0
                  |||||||+-- | SPORT1 trx or IRQ1
                  ||||||+--- | BDMA
                  |||||+---- | IRQE
                  ||||+----- | SPORT0 rec
                  |||+------ | SPORT0 trx
                  ||+------- | IRQL0
                  |+-------- | IRQL1
                  +--------- | IRQ2
            }
#endif

#if CODEC_1847
{************  AD 1847 CODEC Initialization  ******************************}
	{ The stat_flag=0 will prevent reinit of AD1847 from SPORT0 TX ints }
	ax0 = 1;                        { set flag, disable tx ints }
	dm(stat_flag) = ax0;
	imask = b#0001000000;           { enable transmit interrupt }
	ax0 = dm(i6, m6);		{ Start interrupt }
	tx0 = ax0;
check_init:
        ax0 = dm (stat_flag);		{ wait for entire init }
        af = pass ax0;			{ buffer to be sent to }
        if ne jump check_init;		{ the codec            }
        ay0 = 2;
check_acih:
        ax0 = dm (rx_buf);          { once initialized, wait for codec }
        ar = ax0 and ay0;           { to come out of autocalibration }
        if eq jump check_acih;      { wait for bit set }
check_aci1:
        ax0 = dm (rx_buf);          { wait for bit clear }
        ar = ax0 and ay0;
        if ne jump check_aci1;
	idle;
        ay0 = 0xbf3f;               { unmute left DAC }
        ax0 = dm (init_cmds + 6);
        ar = ax0 AND ay0;
        dm (tx_buf) = ar;
        idle;
        ax0 = dm (init_cmds + 7);   { unmute right DAC }
        ar = ax0 AND ay0;
        dm (tx_buf) = ar;
        idle;
	ax0 = 0xc901;			{ Clear autocalibration request }
	dm(tx_buf)=ax0;
	idle;
        ax0 = 0x8000;               { control word to clear over-range flags }
        dm (tx_buf) = ax0;
        idle;

        ifc = b#00000011111111;     { clear any pending interrupt }
        nop;
        imask = b#0000110000;       { enable rx0 interrupt (& irqe) }
#endif
	rts;

{***********************************************************************
 *                              window()                               *
 *  Multiplies time series by window function to reduce sidelobes      *
 *  due to end discontinuities (de-Gibbing).                           *
 ***********************************************************************}
window:	ar=dm(do_window);
	none=pass ar;
	if eq rts;			{ 0=No windowing }

	ar=ar-1;
	i4=^tukey_win;			{ 1=Tukey 25% window }
	if eq jump win1;

	ar=ar-1;
	i4=^ham_win;			{ 2=Hamming window }
	if eq jump win1;

	i4=^bh4_win;			{ 3=Blackman-harris 92 window }
win1:	m4=1; l4=0;
	i0=dm(dataptr); m0=1; l0=0;	{ Ptr to last data block filled }
	i1=dm(dataptr); m1=1; l1=0;
	mx0=dm(i0, m0), my0=pm(i4, m4);
	cntr=N-1;
	do win2 until ce;
	  mr=mx0*my0(rnd), mx0=dm(i0, m0), my0=pm(i4, m4);
win2:	  dm(i1, m1)=mr1;
	mr=mx0*my0(rnd);
	dm(i1, m1)=mr1;
	rts;
  
{***************************************************************************
 * UART routine to send a block of byte size data.
 * Called 9600 times per second. Returns
 * if no data left in block with UART_tx_ch_count=0.  Pointer
 * to data block is UART_tx_ptr and UART_tx_ch_count is block length.
 * only least significant 8 bits are sent, 8N1.           RSL 8 June 96
 *
 * Note: Start bit is logic 0, RS232 high
 *	 Data bits are sent with RS232 high = logic 0
 *	 Stop bit is logic 1, RS232 low
 *	 Output is left logic 1, RS232 low, when no data is being sent
 *
 *	 Arrive with dm(UART_tx_cur_bit)=10     --> Start bit
 *					=9 to 2 --> Data bits
 *					=1      --> Stop bit
 * Cycles: 9 for start, 14 for data bits, 
 ***************************************************************************}
out_block_byte:
	ay0=dm(UART_tx_cur_bit);	{ Are we in a byte and where? }
	ar=ay0-1;			{ Update and set flags }
	if le jump stop_bit;		{ End of byte, it was 1 }
	dm(UART_tx_cur_bit)=ar;		{ Save for next time }
	ay0=9;
	ar=ar-ay0;			{ Was it 10? }
	if eq jump start_bit;
	sr1=dm(UART_tx_cur_data);	{ Get current data }
	sr=lshift sr1 by -1(hi);	{ Move current bit into SR0 sign bit }
	dm(UART_tx_cur_data)=sr1;	{ Put shifted word back }
	ar=pass sr0;			{ Set flags }
	if ge reset flag_out;		{ Send a zero }
	if lt set flag_out;		{ Send a one }
	rts;
start_bit:
	reset flag_out;			{ Start bit = 0 }
	rts;
stop_bit:
	set flag_out;			{ Stop bit = 1 }
	ax0=dm(UART_tx_esc);		{ Was last char an esc ch? If }
	ar=pass ax0;			{   so, skip on by and send }
	if ne jump stop_esc;		{   a second esc ch }
	ay0=dm(UART_tx_resync_count);	{ Every 1000 CH, level=1 for 12 bits } 
	ar=ay0-1;			{ Decrement }
	if eq jump new_sync;		{ 1000 are up, set to 1012 }
	dm(UART_tx_resync_count)=ar;	{ Put back for next time }
	ay0=1000;
	ar=ar-ay0;
	if gt rts;			{ If >1000 do nothing }
	ay0=dm(UART_tx_ch_count);	{ Another character in block? }
	ar=ay0-1;
	if lt rts;			{ Nothing to send, stop decrementing }
	dm(UART_tx_ch_count)=ar;	{ Decremented count back, if >=0 }
	if eq rts;
	ay0=10;
	dm(UART_tx_cur_bit)=ay0;	{ Ready to send a new byte, next call }
	i0=dm(UART_tx_ptr); m0=1; l0=0;	{ Indirect to next data byte }
	ax0=dm(i0, m0);			{ Bump pointer }
{ Put stuff for sending words here, UART_tx_word_len and UART_tx_byte num <<< }
	dm(UART_tx_ptr)=i0;		{ Save bumped pointer }
	m0=0;  ax0=dm(i0, m0);		{ Get next byte (word) }
	ay0=ESC_CH;
	ar=ax0-ay0;
	dm(UART_tx_cur_data)=ax0;	{ Save new data }
	if eq jump start_esc;
	rts;
new_sync:
	ax0=1012;
	dm(UART_tx_resync_count)=ax0;	{ Skips a character every 1000 }
	rts;
start_esc:
	ax0=1;
	dm(UART_tx_esc)=ax0;		{ Flag that extra ESC_CH is being sent}
	rts;
stop_esc:
	ax0=0;
	dm(UART_tx_esc)=ax0;		{ Restore flag to "no esc ch" }
	ax0=ESC_CH;			{ Send a second esc ch }
	dm(UART_tx_cur_data)=ax0;
	ax0=10;
	dm(UART_tx_cur_bit)=ax0;	{   starting next call }
	rts;

{***************************************************************************
 * UART routine to receive characters, 9600 Baud, 8N1.
 * Called 5*9600=48000 times per second.  Parts are from AD timer based UART.
 * All received sequences are 6 bytes starting with an escape character
 * followed by a command character (0 to 255 but not ESC_CH) and four bytes
 * making up two 16 bit words: LS1, MS1, LS2, MS2.  These are placed into a
 * five byte buffer that is checked 9600 times per sec (and thus emptied
 * when full.) Check is in block 5 of the SPORT0 Receive ISR.
 *          RSL 22 June 96
 ***************************************************************************}
uart_rcvr:
	ay0=dm(UART_rx_cur_bit);	{ Zero if not receiving data }
	none=pass ay0;
	if eq jump rx_start;		{ Jump if zero at call, look for start}
	ay1=dm(UART_rx_ctr);		{ Decrement timer ctr and test to see }
	ar=ay1-1; 			{ if bit period has been reached. }
	dm(UART_rx_ctr)=ar;		{ Save }
	if ne rts;
	ax0=1;				{ ay0 still has cur_bit }
	ar=ax0-ay0;			{ Test for stop bit, i.e. cur_bit=1 }
	if eq jump rx_stop_bit;
	ax0=5;				{ Reset ctr to 5 for next bit wait }
	dm(UART_rx_ctr)=ax0;
	ar=ay0-1;			{ Decrement cur_bit }
	dm(UART_rx_cur_bit)=ar;		{ Save for next bit }
	ay0=0x0100;
	ar=dm(UART_rx_cur_byte);
	if not flag_in jump pad_zero;	{ Test rx input bit }
	ar=ar+ay0;			{ Add in a 1 if hi }
pad_zero:				{ Bit was a zero }
	sr=lshift ar by -1 (lo);        { Shift down to ready for next bit }
	dm(UART_rx_cur_byte)=sr0;
	rts;
rx_start:				{ Here looking for start bit }
	if flag_in jump rs1;		{ Test for start bit; return if none }
	ax0=9;
	dm(UART_rx_cur_bit)=ax0;
	ax0=0;
	dm(UART_rx_cur_byte)=ax0;	{ Clear out rcv register}
	ax0=7;				{ Sample at 5x so this will be }
	dm(UART_rx_ctr)=ax0;		{   centered on 1st data bit }
rs1:	rts;
	
rx_stop_bit:				{ Now time for stop bit }
	if flag_in jump rsb1;		{ Jump if stop bit is OK }

	{ A command error has occurred, stop bit not there; process it }
	ax0=-1;
	dm(serial_timeout)=ax0;		{ Stop further errors }
	ax0=0;
	dm(UART_rx_cmd_count)=ax0;	{ Cancel any command fragments }
	dm(UART_rx_cur_bit)=ax0;	{ Start new byte }
	ax1=^cmd_buf;			{ Reset buffer pointer }
	dm(UART_rx_cmd_buf_ptr)=ax1;	{ Ready to receive command }

	ar=dm(serial_error);		{ To be sent to PC at next send }
	ar=ar+1;			{ Bump error count }
	ay0=0X001F;			{ Clear error code }
	ar=ar and ay0;
	ay0=0X0080;			{ Code for stop-bit error }
	ar=ar or ay0;
	dm(serial_error)=ar;		{ Put back to send to PC }
	jump rsb2;
rsb1:	ax0=dm(UART_rx_cur_byte);	{ This is the received byte }
	ay1=dm(UART_rx_cmd_count);	{ Count =0 if waiting for ESC_CH }
	none=pass ay1;
	if eq jump rsb3;		{ Looking for ESC_CH }
	ar=ay1+1;			{ Here to add another ch to cmd_buf }
	dm(UART_rx_cmd_count)=ar;
	ax1=6;
	ar=ax1-ay1;
	if lt rts;			{ This is error, cmd_buf wasn't read }
	i0=dm(UART_rx_cmd_buf_ptr);
	m0=1; l0=0;
	dm(i0, m0)=ax0;			{ Put new byte into cmd_buf }
	dm(UART_rx_cmd_buf_ptr)=i0;	{ Update pointer }
	{ Reset serial_timeout each time a command byte is received }
	{ 20 -> 2ms.  Therefore 10000 -> 1 second. }
	ax0=10000;				{ This is 2 msec worth }
	dm(serial_timeout)=ax0;
rsb2:	ax0=0;				{ Set cur_bit to zero }
	dm(UART_rx_cur_bit)=ax0;
	rts;
rsb3:	ay0=ESC_CH;			{ Ready to start cmd, ch in ax0 }
	none=ax0-ay0;
	if ne jump rsb2;		{ Not ESC_CH so continue to wait }
	ar=ay1+1;			{ Increment count }
	dm(UART_rx_cmd_count)=ar;	{   and save }
	ax1=^cmd_buf;
	dm(UART_rx_cmd_buf_ptr)=ax1;	{ Ready to receive command }
	{ Reset serial_timeout each time a command byte is received }
	ax0=10000;				{ This is 2 msec worth }
	dm(serial_timeout)=ax0;
	jump rsb2;			{ Don't do anything with the ch }

{*************************************************************************
 *       Sine Approximation:     Y = Sin(x)                              *
 *       Calling Parameters:     AX0 = x in scaled 1.15 format           *
 *       Return Values:          AR = y in 1.15 format                   *
 *       Altered Registers:      AY0,AF,AR,MY1,MX1,MF,MR,SR,I4,M4,L4     *
 *       Computation Time:       25 cycles                               *
 * Thanks to Analog Devices Applications #1
 *************************************************************************}

sin:    M4=1;    L4=0;
        I4=^sin_coeff;                  { Pointer to coeff. buffer }
        AY0=H#4000;


        AR=AX0, AF=AX0 AND AY0;         { Check 2nd or 4th quad. }
        IF NE AR=-AX0;                  { If yes, negate input }
#if 0
  {<<<<<<<<<< BOB Change AF to NONE??? }
	{ Change to input in AR (not AX0) Rev 2.A 11/2004 }
        ax0=ar, af=ar AND AY0;                { Check 2nd or 4th quad. }
        IF NE AR=-ar;                   { If yes, negate input }
#endif

        AY0=H#7FFF;
        AR=AR AND AY0;                  { Remove sign bit }
        MY1=AR;
        MF=AR*MY1 (RND), MX1=PM(I4,M4); { MF = x**2 }
        MR=MX1*MY1 (SS), MX1=PM(I4,M4); { MR = C1*x }
        CNTR=3;
        DO approx UNTIL CE;
                MR=MR+MX1*MF (SS);
approx:         MF=AR*MF (RND), MX1=PM(I4,M4);
        MR=MR+MX1*MF (SS);
        SR=ASHIFT MR1 BY 3 (HI);
        SR=SR OR LSHIFT MR0 BY 3 (LO);  { Convert to 1.15 format }
        AR=PASS SR1;
        IF LT AR=PASS AY0;              { Saturate if needed }
        AF=PASS AX0;
        IF LT AR=-AR;                   { Negate output if needed }
        RTS;

{****************************************************************************
* Arctangent Approximation	y   = arctan(q/i)
* Calling Parameters		ax1 = i in 1.15 format
*				ar  = q in 1.15
* Return			ar = arctan(q/i) in scaled 1.15 format
*				(-1.0 for -180 degrees, 1.0 for 180 degrees)
*				and quadrant in SR1 by bit position =1, i.e.,
*				2=q1. 4=q2, 8=q3, 16=q4.
* Computation Time		63 cycles maximum
* Accuracy			Better than 13 bits plus sign
* Errors			i=8000h, q=7FFFh or reverse gives .25 error
*
* Adapted from Analog Devices App Hndbk 1.  RSL
* First separate into quadrants stored in SR1.  Then find arctan for i and q
* both positive.  Here use either q/i or i/q to make input <=1.0.  Finally
* sort out the results to put into the correct octant.
*****************************************************************************}
arctan:	m4=1;    l4=0;
	i4 = ^atn_coeff;		{ Point to coefficients }
	none = pass ax1;		{ Check sign of i }
	if ge jump iplus;
	none = pass ar;			{ Check sign of q }
	sr1 = 4;			{ Bit 2 ==> Quadrant 2, i- q+ }
	if ge jump calc_atan;
	sr1 = 8;			{ Bit 3 ==> Quadrant 3, i- q- }
	jump calc_atan;
iplus:	none = pass ar;
	sr1 = 2;			{ Arrive with flags on ay0 }
	if ge jump calc_atan;		{ Bit 1 ==> Quadrant 1, i+ q+ }
	sr1 = 16;			{ Bit 4 ==> Quadrant 4, i+ q- }
calc_atan:
	ar = abs ar;			{ Now make i+ q+ }
	ay1 = ar;
	ar = abs ax1;
	none = ar-ay1;			{ Find larger for denominator }
	sr0 = 1;			{ Octant: sr0=1 for |i| > |q|  }
	if ge jump atn1;		{ ar >= ay1 }
	ax0 = ay1;			{ Bigger }
	ay1 = ar;			{ Smaller }
	sr0 = 0;			{ Octant: sr0=0 for |q| > |i|  }
	jump atn2;
atn1:	ax0 = ar;			{ Bigger, smaller already in ay1 }
atn2:	ay0 = 0;
	{Here with ay1=msw of numerator, ay0=lsw of numerator, ax0=denom }
        DIVS AY1,AX0;
        DIVQ AX0; DIVQ AX0; DIVQ AX0;
        DIVQ AX0; DIVQ AX0; DIVQ AX0;
        DIVQ AX0; DIVQ AX0; DIVQ AX0;
        DIVQ AX0; DIVQ AX0; DIVQ AX0;
        DIVQ AX0; DIVQ AX0; DIVQ AX0;
        AR=AY0;				{ Qoutient in ay0 }
	MY0=AR;
        MF=AR*MY0 (RND), MY1=pm(i4, m4);
        MR=AR*MY1 (SS), MX1=pm(i4, m4);
        CNTR=3;
        DO atn3 UNTIL CE;
	    MR=MR+MX1*MF (SS), MX1=pm(i4, m4);
atn3:	    MF=AR*MF (RND);
        MR=MR+MX1*MF (SS);
	ar = mr1;
	{ Now have atan(|ay1| / |ax0|) in mr1.  Sort out quadrants: }
        AY0 = H#4000;			{ Ninty degrees }
	none = pass sr0;		{ Octant: sr0=0 for |q| > |i| }
	if gt jump atn4;		{ Go process |i| > |q| }
	af = tstbit 1 of sr1;
	if gt jump atn31;		{ Quadrant 1 and |q| > |i| }
	af = tstbit 2 of sr1;
	if gt jump atn32;		{ Quadrant 2 and |q| > |i| }
	af = tstbit 3 of sr1;
	if gt jump atn33;		{ Quadrant 3 and |q| > |i| }
	{ Here for quadrant 4 and |q| > |i| }
	ar = ar - ay0;
	rts;
atn31:	ar = ar - ay0;
	ar = -ar;
	rts;
atn32:	ar = ar + ay0;
	rts;
atn33:	ar = ar + ay0;
	ar = -ar;
	rts;
atn4:	ay1 = H#7FFF;			{ 180 degrees }
	af = tstbit 1 of sr1;
	if gt jump atn41;		{ Quadrant 1 and |i| > |q| }
	af = tstbit 2 of sr1;
	if gt jump atn42;		{ Quadrant 2 and |i| > |q| }
	af = tstbit 3 of sr1;
	if gt jump atn43;		{ Quadrant 3 and |i| > |q| }
	{ Here for quadrant 4 and |i| > |q| }
	ar = -ar;
	rts;
atn41:	rts;
atn42:	ar = ar - ay1;
	ar = -ar;
	rts;
atn43:	ar = ar - ay1;
	rts;

{**************************************************************************
 * Hilbert Transforms, 1 and 2
 * Inputs:	ax0 = new received signal data for delay
 *		ay0 = new received signal for 90 deg rel phase shift
 * Outputs:	ax1 = delayed signal data
 *		mr1 = delayed and 90 deg phase shifted signal data
 * Instruction cycles:  22 + H1M1ON2  (57 cycles for 71 point Hilbert )
 *                    RSL                   17 June 96
 **************************************************************************}
do_hilbert1:				{ 48 KHz Hilbert }
	dm(m1_sav) = m1;
	m1 = 1;

	i0 = dm(h1delay_i0_sav);  m0=0;  l0=%h1delay;
	ax1 = dm(i0, m0);		{ get ax1, the delayed output }
	dm(i0, m1) = ax0;		{ Put new data in, update pointer }
	dm(h1delay_i0_sav) = i0;	{ Save pointer }

	i0=dm(h1data_i0_sav); m0=2; l0=%h1data;  { i0 points to data }
	i4=^hilbert1_coeff; m4=1; l4=%hilbert1_coeff;
	dm(i0, m1)=ay0;			{ Enter new data and bump pointer 1 }
	mr=0, mx0=dm(i0, m0), my0=pm(i4, m4);
	cntr=H1M3ON2;
	do hil1_loop until ce;
hil1_loop:   mr=mr+mx0*my0(SS), mx0=dm(i0, m0), my0=pm(i4, m4);
	mr=mr+mx0*my0(SS), mx0=dm(i0, m1), my0=pm(i4, m4);
	mr=mr+mx0*my0(RND);			{ mr1 = phase shifted output }
	dm(h1data_i0_sav)=i0;

	m1 = dm(m1_sav);
	rts;

do_hilbert2:
	dm(m1_sav) = m1;
	m1 = 1;

	i0 = dm(h2delay_i0_sav);  m0=0;  l0=%h2delay;
	ax1 = dm(i0, m0);		{ get ax1, the delayed output }
	dm(i0, m1) = ax0;		{ Put new data in, update pointer }
	dm(h2delay_i0_sav) = i0;	{ Save pointer }

	i0=dm(h2data_i0_sav);  m0=2;  l0=%h2data;  { i0 points to data }
	{ For now both 1 and 2 are same 71 point Hilbert, ie, hilbert1_coeff }
	i4=^hilbert1_coeff; m4=1; l4=%hilbert1_coeff;
	dm(i0, m1)=ay0;			{ New data and bump pointer }
	mr=0, mx0=dm(i0, m0), my0=pm(i4, m4);
	cntr=H1M3ON2;
	do hil2_loop until ce;
hil2_loop:   mr=mr+mx0*my0(SS), mx0=dm(i0, m0), my0=pm(i4, m4);
	mr=mr+mx0*my0(SS), mx0=dm(i0, m1), my0=pm(i4, m4);
	mr=mr+mx0*my0(RND);			{ mr1 = phase shifted output }
	dm(h2data_i0_sav)=i0;

	m1 = dm(m1_sav);
	rts;

{ Hilbert 3 - a little different.  Uses HIL_3_48.DAT running at 48 KHz in order
  to get response down to 300 Hz.  Both the delayed and delayed+phase shifted
  inputs are the same, and here come in mr1.  Outputs are mr1 and ax1.    }
do_hilbert3:				{ 48 KHz Hilbert for transmitting }
	dm(m1_sav) = m1;
	m1 = 1;

	i0 = dm(h3delay_i0_sav);  m0=0;  l0=%h3delay;
	ax1 = dm(i0, m0);		{ get ax1, the delayed output }
	dm(i0, m1) = mr1;		{ Put new data in, update pointer }
	dm(h3delay_i0_sav) = i0;	{ Save pointer }

	i0=dm(h3data_i0_sav); m0=2; l0=%h3data;  { i0 points to data }
	i4=^hilbert3_coeff; m4=1; l4=%hilbert3_coeff;
	dm(i0, m1)=mr1;			{ Enter new data and bump pointer 1 }
	mr=0, mx0=dm(i0, m0), my0=pm(i4, m4);
	cntr=H3M3ON2;
	do hil3_loop until ce;
hil3_loop:   mr=mr+mx0*my0(SS), mx0=dm(i0, m0), my0=pm(i4, m4);
	mr=mr+mx0*my0(SS), mx0=dm(i0, m1), my0=pm(i4, m4);
	mr=mr+mx0*my0(RND);			{ mr1 = phase shifted output }
 if mv sat mr;
	dm(h3data_i0_sav)=i0;
	m1 = dm(m1_sav);
	rts;

{ Hilbert 4 - Same as 3, but has independent inputs and outputs
  for  delayed and phase shifted paths. Shares dm with hilbert3.
  Uses HIL_3_48.DAT running at 48 KHz in order to get response down to 300 Hz.
  Delayed path: ay1 in, ax1 out;  90 deg path: mr1 in, mr1 out.    }
do_hilbert4:				{ 48 KHz Hilbert for receiving }
	dm(m1_sav) = m1;
	m1 = 1;

	i0 = dm(h3delay_i0_sav);  m0=0;  l0=%h3delay;
	ax1 = dm(i0, m0);		{ get ax1, the delayed output }
	dm(i0, m1) = ay1;		{ Put new data in, update pointer }
	dm(h3delay_i0_sav) = i0;	{ Save pointer }

	i0=dm(h3data_i0_sav); m0=2; l0=%h3data;  { i0 points to data }
	i4=^hilbert3_coeff; m4=1; l4=%hilbert3_coeff;
	dm(i0, m1)=mr1;			{ Enter new data and bump pointer 1 }
	mr=0, mx0=dm(i0, m0), my0=pm(i4, m4);
	cntr=H3M3ON2;
	do hil4_loop until ce;
hil4_loop:   mr=mr+mx0*my0(SS), mx0=dm(i0, m0), my0=pm(i4, m4);
	mr=mr+mx0*my0(SS), mx0=dm(i0, m1), my0=pm(i4, m4);
	mr=mr+mx0*my0(RND);			{ mr1 = phase shifted output }
	if mv sat mr;
	dm(h3data_i0_sav)=i0;

	m1 = dm(m1_sav);
	rts;

{*****************************************************************************
 * FIR1   A 2.8 kHz LPF with 80 dB down above 4KHz.  48 kHz sample rate
 *        Note - These FIRs have a gain of 4 that is removed after summing.
 * Inputs: Assumes input array has been updated and pointer is
 *            dm(fir1i_i0_sav) or q.
 * Output: Filtered output point in MR1.
 * Cycles: 13 + TAPS1  (143 cycles for TAPS1=130)
 *****************************************************************************}
do_fir1i:
	ax0=dm(fft_fcn);		{ See if wider filter is being used }
	ar=tstbit 7 of ax0;
	if ne jump f1i_a;		{ Yes it is }
	i0=dm(fir1i_i0_sav);  m0=1;  l0=%idata;
        i4=^fir1_coefs;      m4=1;  l4=%fir1_coefs;
        cntr=TAPS1-1;                   { 2.8 kHz LPF }
        mr=0, mx0=dm(i0,m0), my0=pm(i4,m4);
        do fir1iloop until ce;
fir1iloop:   mr=mr+mx0*my0(ss), mx0=dm(i0,m0), my0=pm(i4,m4);
        mr=mr+mx0*my0(rnd);		{ Filtered data in MR1 }
	{ Gain of fir, needs to be removed now }
	sr = ashift mr2 by -2 (hi);
	sr = sr or lshift mr1 by -2 (lo);
	mr2=sr1; mr1=sr0;
	mx0=0; my0=0; mr=mr+mx0*my0 (ss);
	if mv sat mr;
        dm(fir1i_i0_sav)=i0;		{ i0 for next call of do_fir1 }
	rts;

	{ Same filter again, but with 4 kHz LP coefficients }
f1i_a:	i0=dm(fir1i_i0_sav);  m0=1;  l0=%idata;
        i4=^fir1a_coefs;      m4=1;  l4=%fir1a_coefs;
        cntr=TAPS1-1;                   { 4 kHz LPF }
        mr=0, mx0=dm(i0,m0), my0=pm(i4,m4);
        do f1i_aloop until ce;
f1i_aloop:   mr=mr+mx0*my0(ss), mx0=dm(i0,m0), my0=pm(i4,m4);
        mr=mr+mx0*my0(rnd);		{ Filtered data in mr1 }
	{ Gain of fir, needs to be removed now }
	sr = ashift mr2 by -2 (hi);
	sr = sr or lshift mr1 by -2 (lo);
	mr2=sr1; mr1=sr0;
	mx0=0; my0=0; mr=mr+mx0*my0 (ss);
	if mv sat mr;
        dm(fir1i_i0_sav)=i0;		{ i0 for next call of do_fir1 }
	rts;

do_fir1q:
	ax0=dm(fft_fcn);		{ See if wider filter is being used }
	ar=tstbit 7 of ax0;
	if ne jump f1q_a;		{ Yes it is }
	i0=dm(fir1q_i0_sav);  m0=1;  l0=%qdata;
        i4=^fir1_coefs;      m4=1;  l4=%fir1_coefs;
        cntr=TAPS1-1;                   { 2.8 kHz LPF }
        mr=0, mx0=dm(i0,m0), my0=pm(i4,m4);
        do fir1qloop until ce;
fir1qloop:   mr=mr+mx0*my0(ss), mx0=dm(i0,m0), my0=pm(i4,m4);
        mr=mr+mx0*my0(rnd);		{ Filtered data in MR1 }
	if mv sat mr;
	{ Gain of fir needs to be removed now }
	sr = ashift mr2 by -2 (hi);
	sr = sr or lshift mr1 by -2 (lo);
	mr2=sr1; mr1=sr0;
	mx0=0; my0=0; mr=mr+mx0*my0 (ss);
	if mv sat mr;
        dm(fir1q_i0_sav)=i0;		{ i0 for next call of do_fir1 }
	rts;

	{ Same filter again, but with 4 kHz LP coefficients }
f1q_a:	i0=dm(fir1q_i0_sav);  m0=1;  l0=%qdata;
        i4=^fir1a_coefs;      m4=1;  l4=%fir1a_coefs;
        cntr=TAPS1-1;                   { 4 kHz LPF }
        mr=0, mx0=dm(i0,m0), my0=pm(i4,m4);
        do f1q_aloop until ce;
f1q_aloop:   mr=mr+mx0*my0(ss), mx0=dm(i0,m0), my0=pm(i4,m4);
        mr=mr+mx0*my0(rnd);		{ Filtered data in mr1 }
	{ Gain of fir, needs to be removed now }
	sr = ashift mr2 by -2 (hi);
	sr = sr or lshift mr1 by -2 (lo);
	mr2=sr1; mr1=sr0;
	mx0=0; my0=0; mr=mr+mx0*my0 (ss);
	if mv sat mr;
        dm(fir1q_i0_sav)=i0;		{ i0 for next call of do_fir1 }
	rts;

{*************************************************************************
 * INTERP_FILTER  A 3.7 kHz FIR LPF with 65 dB down above 4.8KHz.  Allows
 *   output DAC to run at 48 KHz.  Better would be a 9.6 kHz DAC, but
 *   the AD1847 ADC and DAC share a common rate.
 *   Note that fir coefs provide a gain of 4, compensating for interpolation
 *   loss of 5 due to added zeros (almost).
 *   Added wider  4kHz option with less rejection (24 Oct 00)
 * Inputs: Data has already been updated
 * Output: Filtered output point in MR1
 * Cycles: 13 + TAPS2  (143 cycles for TAPS2=130)
 *****************************************************************************}
interp_filter:
	ax0=dm(fft_fcn);		{ See if wider filter is being used }
	ar=tstbit 13 of ax0;
	if ne jump interpf_a;		{ Yes it is }

        i0=dm(fir2_i0_sav); m0=1; l0=TAPS2;
        i4=^fir1_coefs; m4=1; l4=TAPS2;
        cntr=TAPS2-1;
        mr=0, mx0=dm(i0,m0), my0=pm(i4,m4);
        do fir2loop_a until ce;
fir2loop_a:   mr=mr+mx0*my0(ss), mx0=dm(i0,m0), my0=pm(i4,m4);
        mr=mr+mx0*my0(rnd);
        if mv sat mr;
	dm(fir2_i0_sav)=i0;
	rts;				{ Filtered sample in mr1 }

	{ Same filter again, but with 4 kHz coefficients }
interpf_a:
        i0=dm(fir2_i0_sav); m0=1; l0=TAPS2;
        i4=^fir1a_coefs;      m4=1;  l4=%fir1a_coefs;
        cntr=TAPS2-1;
        mr=0, mx0=dm(i0,m0), my0=pm(i4,m4);
        do fir2loop until ce;
fir2loop:   mr=mr+mx0*my0(ss), mx0=dm(i0,m0), my0=pm(i4,m4);
        mr=mr+mx0*my0(rnd);
        if mv sat mr;
	dm(fir2_i0_sav)=i0;
	rts;				{ Filtered sample in mr1 }

{*************************************************************************
 * AUDIO_FIR  A general purpose FIR filter for audio filtering.
 * The coefficients for narrow-band filters will result in some gain
 * for sine-wave inputs. If desired, all or part of this gain can be
 * removed by making audio_fir_gain=-1, -2, ...-8. These are powers of
 * two that produce a shift in the output.
 * The fir coefficients come from a pointer list fir3ptr that is indexed
 * by dm(audio_fir_select).
 *
 * Inputs: New input point in AR
 * Output: Filtered output point in MR1
 * Cycles: 24 + TAPS3  (224 cycles for TAPS3=199)
 *****************************************************************************}
audio_fir:
        i0=dm(fir3_i0_sav); m0=1; l0=TAPS3;
	dm(i0,m0) = ar;
	l4=0;				{ Make sure that we don't wrap }
	i4=^fir3ptr;			{ Base for ptr table for coefs }
	m4=dm(audio_fir_select);	{ Index into ptr table }
	modify(i4, m4);			{ Move into table } 
	ax0=pm(i4, m4);			{ They don't allow 'i4=' }
	i4=ax0;
	m4=1; l4=TAPS3;
        mr=0, mx0=dm(i0,m0), my0=pm(i4,m4);
        cntr=TAPS3-1;
        do fir3loop until ce;
fir3loop:   mr=mr+mx0*my0(ss), mx0=dm(i0,m0), my0=pm(i4,m4);
        mr=mr+mx0*my0(rnd);

	se=dm(audio_fir_gain);		{ 0 to -8 }
	sr = ashift mr2 (hi);		{ get any bits that ended up in mr2 }
	sr = sr or lshift mr1 (lo);	{ Divide the low order bits }
	mr1 = sr0;			{ Get ready to saturate }
	mr2 = sr1;			{ Must follow mr1=... }
	mx0=0;				{ Fake a multiply }
	mr=mr+mx0*my0 (ss);		{ Set the mv status flag }
	if mv sat mr;			{ Take care of overflow }
	dm(fir3_i0_sav)=i0;		{ Save for next time }
	rts;

{*****************************************************************************
 * FIR_XMT_AUDIO   A 2.8 kHz LPF with -80dB  above 4KHz.  48 kHz sample rate
 * Used to shape input audio for ssb transmission.
 * Note: This filter has a gain of 2; x1/2 is from leading shift and x4
 * due to coefficient scaling.
 * There is no saturation provision. The signal levels must be arranged
 * to never reach full levels as this would cause severe splatter,
 * which is part of why this filter is used (in SSB or AM).
 * Inputs: New data point in AR, pointer is dm(fir4_i0_sav).
 * Output: Filtered output point in MR1.
 * Cycles: 13 + TAPS4  (143 cycles for TAPS4=130)
 *****************************************************************************}
fir_xmt_audio:
	sr=ashift ar by -1 (hi);	{ Times one half }
	ar=sr1;
	i0=dm(fir4_i0_sav);  m0=1;  l0=%xmt_audio_data;
	dm(i0,m0) = ar;			{ Update delay line }
        i4=^fir4_coefs;      m4=1;  l4=%fir4_coefs;
        cntr=TAPS4-1;
        mr=0, mx0=dm(i0,m0), my0=pm(i4,m4);
        do fir4loop until ce;
fir4loop:   mr=mr+mx0*my0(ss), mx0=dm(i0,m0), my0=pm(i4,m4);
        mr=mr+mx0*my0(rnd);		{ Filtered data in MR1 }
        dm(fir4_i0_sav)=i0;		{ i0 for next call of fir_xmt_audio }
	rts;

{*****************************************************************************
 * FIR_XMT_CW LPF.  48 kHz sample rate
 * Used to shape keying signal for cw transmission.
 * Inputs: Data in AR and pointer is dm(fir5_i0_sav).
 * Output: Filtered output point in MR1.
 * Cycles: 13 + TAPS5  (213 cycles for TAPS5=200)
 *****************************************************************************}
fir_xmt_cw:
	i0=dm(fir5_i0_sav);  m0=1;  l0=%xmt_cw_data;
	dm(i0,m0) = ar;			{ Update delay line }
        i4=^fir5_coefs;      m4=1;  l4=%fir5_coefs;
        cntr=TAPS5-1;
        mr=0, mx0=dm(i0,m0), my0=pm(i4,m4);
        do fir5loop until ce;
fir5loop:   mr=mr+mx0*my0(ss), mx0=dm(i0,m0), my0=pm(i4,m4);
        mr=mr+mx0*my0(rnd);		{ Filtered data in MR1 }
	{ Gain of fir needs to be removed now }
	sr = ashift mr2 by -5 (hi);
	sr = sr or lshift mr1 by -5 (lo);
	mr1 = sr0;
        dm(fir5_i0_sav)=i0;		{ i0 for next call of fir_xmt_cw }
	rts;

{*****************************************************************************
 * FIR10HP HPF.  48 kHz sample rate
 * Used to restrict input bw for fm detector.
 * Inputs: Data in mx0 and pointer is dm(fir6_i0_sav).
 * Output: Filtered output point in mr1.
 * Cycles: 13 + TAPS6  (50 cycles for TAPS6=37)
 *****************************************************************************}
fir10hp:
	i0=dm(fir6_i0_sav);  m0=1;  l0=%rcv_fm;
	dm(i0,m0) = mx0;			{ Update delay line }
        i4=^fir6_coefs;      m4=1;  l4=%fir6_coefs;
        cntr=TAPS6-1;
        mr=0, mx0=dm(i0,m0), my0=pm(i4,m4);
        do fir6loop until ce;
fir6loop:   mr=mr+mx0*my0(ss), mx0=dm(i0,m0), my0=pm(i4,m4);
        mr=mr+mx0*my0(rnd);		{ Filtered data in mr1 }
        dm(fir6_i0_sav)=i0;		{ i0 for next call of fir10hp }
	rts;

{*****************************************************************************
 * FIR6LP1 LPF.  48 kHz sample rate
 * Used to extract I component of received carrier for fm detector.
 * Inputs: Data in mr1 and pointer is dm(fir7_1_i0_sav).
 * Output: Filtered output point in MR1.
 * Cycles: 13 + TAPS6  (43 cycles for TAPS7=30)
 *****************************************************************************}
fir6lp1:
	i0=dm(fir7_1_i0_sav);  m0=1;  l0=%rcv1_fm;
	dm(i0,m0) = mr1;			{ Update delay line }
        i4=^fir7_coefs;      m4=1;  l4=%fir7_coefs;
        cntr=TAPS7-1;
        mr=0, mx1=dm(i0,m0), my1=pm(i4,m4);
        do fir7_1loop until ce;
fir7_1loop:   mr=mr+mx1*my1(ss), mx1=dm(i0,m0), my1=pm(i4,m4);
        mr=mr+mx1*my1(rnd);		{ Filtered data in mr1 }
        dm(fir7_1_i0_sav)=i0;		{ i0 for next call of fir6lp1 }
	rts;

{*****************************************************************************
 * FIR6LP2 LPF.  48 kHz sample rate
 * Used to extract Q component of received carrier for fm detector.
 * Inputs: Data in mr1 and pointer is dm(fir7_2_i0_sav).
 * Output: Filtered output point in mr1.
 * Cycles: 13 + TAPS6  (43 cycles for TAPS7=30)
 *****************************************************************************}
fir6lp2:
	i0=dm(fir7_2_i0_sav);  m0=1;  l0=%rcv2_fm;
	dm(i0,m0) = mr1;			{ Update delay line }
        i4=^fir7_coefs;      m4=1;  l4=%fir7_coefs;
        cntr=TAPS7-1;
        mr=0, mx1=dm(i0,m0), my1=pm(i4,m4);
        do fir7_2loop until ce;
fir7_2loop:   mr=mr+mx1*my1(ss), mx1=dm(i0,m0), my1=pm(i4,m4);
        mr=mr+mx1*my1(rnd);		{ Filtered data in mr1 }
        dm(fir7_2_i0_sav)=i0;		{ i0 for next call of fir6lp2 }
	rts;

{*****************************************************************************
 * FIR_FM_XMT_AUDIO   A 3.2 kHz LPF with -80dB  above 5KHz. 48 kHz sample rate
 * Used to shape input audio for FM transmission. Includes preemphasis.
 * Shares data array with FIR_XMT_AUDIO.
 * Note: This filter has a gain of 4 due to coefficient scaling.
 * Inputs: New data point in AR, pointer is dm(fir4_i0_sav).
 * Output: Filtered output point in MR1.
 * Cycles: 13 + TAPS4  (143 cycles for TAPS4=130)
 *****************************************************************************}
fir_fm_xmt_audio:
sr=ashift ar by -2 (hi);
ar=sr1;					{UNTIL RESCALE COEFFICIENTS }
	i0=dm(fir4_i0_sav);  m0=1;  l0=%xmt_audio_data;
	dm(i0,m0) = ar;			{ Update delay line }
        i4=^fir8_coefs;      m4=1;  l4=%fir8_coefs;
        cntr=TAPS4-1;
        mr=0, mx0=dm(i0,m0), my0=pm(i4,m4);
        do fir8loop until ce;
fir8loop:   mr=mr+mx0*my0(ss), mx0=dm(i0,m0), my0=pm(i4,m4);
        mr=mr+mx0*my0(rnd);		{ Filtered data in MR1 }
	if mv sat mr;
{	sr = ashift mr2 by -2 (hi);
	sr = sr or lshift mr1 by -2 (lo);
	mr1 = sr0;		}
        dm(fir4_i0_sav)=i0;		{ i0 for next call of fir_fm_xmt_audio}
	rts;

{*****************************************************************************
 * FIR1200 LPF.  9600 kHz sample rate
 * Used to Nyquist limit the data going to the 2400 sample/sec FFT.
 * Inputs: Data in mr1 and pointer is dm(fir1200_i0_sav).
 * Output: Filtered output point in mr1.
 * Cycles: 13 + TAPS6  (73 cycles for TAPS8=60)
 *****************************************************************************}
fir1200:
	i0=dm(fir1200_i0_sav);  m0=1;  l0=%lp1200;
	dm(i0,m0) = mr1;		{ Update delay line }
        i4=^fir1200_coefs;    m4=1;  l4=%fir1200_coefs;
        cntr=TAPS8-1;
        mr=0, mx1=dm(i0,m0), my1=pm(i4,m4);
        do f1200_loop until ce;
f1200_loop:   mr=mr+mx1*my1(ss), mx1=dm(i0,m0), my1=pm(i4,m4);
        mr=mr+mx1*my1(rnd);		{ Filtered data in mr1 }
        dm(fir1200_i0_sav)=i0;		{ i0 for next call of fir1200 }
	rts;

{*****************************************************************************
 * FIR2400 LPF.  9600 kHz sample rate
 * Used to Nyquist limit the data going to the 4800 sample/sec FFT.
 * Inputs: Data in mr1 and pointer is dm(fir2400_i0_sav).
 * Output: Filtered output point in mr1.
 * Cycles: 13 + TAPS6  (73 cycles for TAPS8=60)
 *****************************************************************************}
fir2400:
	i0=dm(fir2400_i0_sav);  m0=1;  l0=%lp2400;
	dm(i0,m0) = mr1;		{ Update delay line }
        i4=^fir2400_coefs;    m4=1;  l4=%fir2400_coefs;
        cntr=TAPS8-1;
        mr=0, mx1=dm(i0,m0), my1=pm(i4,m4);
        do f2400_loop until ce;
f2400_loop:   mr=mr+mx1*my1(ss), mx1=dm(i0,m0), my1=pm(i4,m4);
        mr=mr+mx1*my1(rnd);		{ Filtered data in mr1 }
        dm(fir2400_i0_sav)=i0;		{ i0 for next call of fir2400 }
	rts;

{*****************************************************************************
 * WFIR1   A 2.8 kHz LPF with 80 dB down above 4KHz.  48 kHz sample rate
 *        Note - LP2_8.DAT has a gain of 4 that is removed after summing.
 * Inputs: mr1
 * Output: Filtered output point in MR1.
 * Cycles: 17 + TAPS1  (147 cycles for TAPS1=130)
 *****************************************************************************}

{>>> CAUTION WFIR USERS: Borrowed for ssb xmit, raised gain x2 (see Q vers) }

wfir1i:
	i0=dm(wfir1i_i0_sav);  m0=1;  l0=%idata;
	dm(i0, m0) = mr1;		{ New input }
        i4=^fir1_coefs;      m4=1;  l4=%fir1_coefs;
        cntr=TAPS1-1;                   { 2.8 kHz LPF }
        mr=0, mx0=dm(i0,m0), my0=pm(i4,m4);
        do wfir1iloop until ce;
wfir1iloop:   mr=mr+mx0*my0(ss), mx0=dm(i0,m0), my0=pm(i4,m4);
        mr=mr+mx0*my0(rnd);		{ Filtered data in MR1 }
	{ Gain of fir, needs to be removed now }
  { >>>THESE WERE "by -2"  }
	sr = ashift mr2 by -1 (hi);
	sr = sr or lshift mr1 by -1 (lo);
	mr2=sr1; mr1=sr0;
	mx0=0; my0=0; mr=mr+mx0*my0 (ss); { Output in mr1 }
	if mv sat mr;
        dm(wfir1i_i0_sav)=i0;		{ i0 for next call of wfir1i }
	rts;

wfir1q:
	i0=dm(fir1q_i0_sav);  m0=1;  l0=%qdata;
	dm(i0, m0) = mr1;		{ New input }
        i4=^fir1_coefs;      m4=1;  l4=%fir1_coefs;
        cntr=TAPS1-1;                   { 2.8 kHz LPF }
        mr=0, mx0=dm(i0,m0), my0=pm(i4,m4);
        do wfir1qloop until ce;
wfir1qloop:   mr=mr+mx0*my0(ss), mx0=dm(i0,m0), my0=pm(i4,m4);
        mr=mr+mx0*my0(rnd);		{ Filtered data in MR1 }
	{ Gain of fir, needs to be removed now }
	sr = ashift mr2 by -2 (hi);
	sr = sr or lshift mr1 by -2 (lo);
	mr2=sr1; mr1=sr0;
	mx0=0; my0=0; mr=mr+mx0*my0 (ss);
	if mv sat mr;			{ Output in mr1 }
        dm(fir1q_i0_sav)=i0;		{ i0 for next call of wfir1 }
	rts;

{*****************************************************************************
 * WFIR2   A 5 kHz LPF.  48 kHz sample rate
 * Inputs: mr1
 * Output: Filtered output point in MR1.
 * Cycles: 13 + TAPS1  (143 cycles for TAPS1=130)
 *****************************************************************************}
wfir2i:
	i0=dm(fir1i_i0_sav);  m0=1;  l0=%idata;
	dm(i0, m0) = mr1;		{ New input }
        i4=^wfir2_coefs;      m4=1;  l4=%wfir2_coefs;
        cntr=TAPS1-1;                   { 5 kHz }
        mr=0, mx0=dm(i0,m0), my0=pm(i4,m4);
        do wfir2iloop until ce;
wfir2iloop:   mr=mr+mx0*my0(ss), mx0=dm(i0,m0), my0=pm(i4,m4);
        mr=mr+mx0*my0(rnd);		{ Filtered data in MR1 }
	if mv sat mr;
        dm(fir1i_i0_sav)=i0;		{ i0 for next call of wfir1i }
	rts;

wfir2q:
	i0=dm(fir1q_i0_sav);  m0=1;  l0=%qdata;
	dm(i0, m0) = mr1;		{ New input }
        i4=^wfir2_coefs;      m4=1;  l4=%wfir2_coefs;
        cntr=TAPS1-1;                   { 5 kHz LPF }
        mr=0, mx0=dm(i0,m0), my0=pm(i4,m4);
        do wfir2qloop until ce;
wfir2qloop:   mr=mr+mx0*my0(ss), mx0=dm(i0,m0), my0=pm(i4,m4);
        mr=mr+mx0*my0(rnd);		{ Filtered data in MR1 }
	if mv sat mr;
        dm(fir1q_i0_sav)=i0;		{ i0 for next call of wfir1 }
	rts;

{*************************************************************************
 *      getrnd    Random Number Generator - 8 uniform rn added together
 *                to approximate Gaussian rn.  RSL 14 Feb 96
 * Uniform routine from AD DSP Appl Vol 1.  The linear congruence method is
 * described in Kunth, D. E. "The Art of Computer Programming, Vol 2", Addison
 * Wesle, 1969.  This method multiplies the last random number by a constant
 * a, adds a second constant c and then takes the most significant 16 bits as
 * the new uniformly distributed random number that is uniform on the full
 * 16 bit range.  Taken as signed 1.15 format number this corresponds to -1.0
 * to +32767/32768 with zero mean.
 * After dividing by 8 to prevent numerical overflow, 8 of these uniform
 * random numbers are added together.  An important mathematical principal,
 * called the Central Limit theorem, tells us that as the number of independent
 * numbers added together gets large the result will closely approximate
 * a Gaussian distribution.  Eight gives a good approximation.
 * The sample output in af is in 1.15 format with sigma=0.20412 (sig^2= 1/24)
 * with mean=0;
 * Thus  peak/sigma is 1/.20412 = 4.9 which is sufficient for 99.99998% of
 * all Gaussian samples.
 *
 * Calling value:  dm(seed_msw), dm(seed_lsw)
 * Return value:   af is noise sample
 * Instructions:   134 instruction cycles per output
 *
 *****************************************************************************}
getrnd:	{ The constant a=1664525 decimal, and is represented here as two 16
	  bit values:       }
        my1=25;                         { Upper half of a (1664525/65536) }
        my0=26125;                      { Lower half of a, the remainder }
        af=pass 0;			{ Clear the arithmetic accumulator }
        cntr=8;				{ The number of uniform rn added }
	{ Now loop 8 times to generate a noise sample:   }
        do randloop until ce;		{ Decrease cntr until 0 }
	  sr1=dm(seed_msw);		{ Get the 32 bit seed from last }
	  sr0=dm(seed_lsw);		{   call to this fcn or last loop }

          mr=sr0*my1(uu);		{ 32 bit multiply: a(hi) * x(lo) }
          mr=mr+sr1*my0(uu);            {   and a(hi) * x(lo) + a(lo) X x(hi) }
          si=mr1;			{ Temp storage }
          mr1=mr0;
          mr2=si;
          mr0=h#fffe;                   { c=32767, left-shifted by 1 }
          mr=mr+sr0*my0(uu);            { (above) + a(lo) X x(lo) + c }
          sr=ashift mr2 by 15 (hi);
          sr=sr or lshift mr1 by -1 (hi); { Right-shift by 1 }
          sr=sr or lshift mr0 by -1 (lo); { Now have uniform rn in sr1 }
	  { Use seed_msw and seed_lsw as temporary storage.  This is as
	    efficient as borrowing some unused registers, and less risky when
	    program changes are made. Leaves next seed where we want it, also:}
	  dm(seed_msw)=sr1;		{ Save new seed value, high 16 bits }
	  dm(seed_lsw)=sr0;		{    and low 16 }
	  { Uniform random number still in sr1.  Add on to accumulator:  }
          sr=ashift sr1 by -3 (hi);	{ Divide by 8, i.e., shift right 3 }
randloop: af=sr1+af;			{ Accumulate 8 uniform rn }
	rts;				{ random 16-bit value in af }

{*********************  ADSP 2181 Initialization  **************************}
init_2181:
	set flag_out;			{ UART transmitter }
        ax0 = 1024;
	dm(fft_buff_count) = ax0;

	ax0 = 0;
	dm(block_count) = ax0;		{ 9.6 kHz block calls }
	dm(UART_tx_buf_select) = ax0;	{ Start sending from buffer 0 }
	dm(mode_select) = ax0;		{ SSB mode }
	dm(UART_tx_data_done) = ax0;	{ Start out sending data }
	dm(audio_fir_on) = ax0;		{ 0=audio_fir() off }
	ax0=8;
	dm(audio_fir_select) = ax0;	{ bit 0=coefs0, gain of 1/8 }
	ax0 = 1;
	dm(get_data_block) = ax0;	{ Start out gathering data }
	dm(do_window) = ax0;
	dm(ssb_select) = ax0;		{ USB }

	ax0 = ^inputreal0;
	dm(dataptr) = ax0;
	ax0 = ^inputreal1;
	dm(inptr0) = ax0;
	ax0 = ^inputreal2;
	dm(inptr1) = ax0;

{***********  UART Transmitter Initialization    ****************************
 *
 *  Currently there is a very simple format:
 *	esc_a		2 Bytes		ESC_CH, '$'
 *	out_log[]	512 Bytes	Output of fft in 8 bit log format
 *	status[]	20 Bytes	Available for status info
 * These are sent sent in this order.
 * Make sure last byte of status_buf != ESC_CH or esc seq won't work.
 ***************************************************************************}
	{ Note - All escape sequences are two characters with ESC_CH first.
	  The data block must be 2 ch or else ESC_CH will be sent twice.  }
	ax0=ESC_CH; dm(esc_a)  =ax0;	{ Initialize escape sequence }
	ax0=0x41;   dm(esc_a+1)=ax0;
	ax0=0x32; dm(status_buf+0)=ax0;	{ Version number }
	ax0=00;	dm(status_buf+1)=ax0;	{ UART Receive errors }
	ax0=32;	dm(status_buf+2)=ax0;	{ Last cmd received }
	{ For the voltage-to-frequency case, the next two bytes are used
	  to return the dB power level driving the v2f.
	  For this, dm(status_buf+3), 4 is save_db_v2f, low then high bytes.}
	ax0=32;	dm(status_buf+3)=ax0;	{ AGC Gain, low }
	ax0=32;	dm(status_buf+4)=ax0;	{ AGC Gain, high }
	ax0=32;	dm(status_buf+5)=ax0;	{ Ave power, low, FM only }
	ax0=32;	dm(status_buf+6)=ax0;	{ Ave power, high, FM only }
	ar=^ptr_table;
	dm(status_buf+7)=ar;		{ General data to PC - see +13, low }
	sr=lshift ar by -8 (hi);
	dm(status_buf+8)=sr1;		{ General data to PC, high }
{ Low 4 bits of +13 are index. If  0 then bytes +7 and +8 are the address
  of the pointer table (default). Else data requested is returned in bytes
  +7 and +8 unless it is from PM in which case the low 8 bits are in +18 }
	ax0=32;	dm(status_buf+9)=ax0;	{ A/D biggest, low }
	ax0=32;	dm(status_buf+10)=ax0;  { A/D biggest, high }
	ax0=32;	dm(status_buf+11)=ax0;	{ D/A biggest, low }
	ax0=32;	dm(status_buf+12)=ax0;	{ D/A biggest, high }
	ax0=32;	dm(status_buf+13)=ax0;  { Index of data in 7+8+18 (low 4 bits)}
	ax0=32;	dm(status_buf+14)=ax0;	{ Modified time tag back to PC }
	ax0=0;	dm(status_buf+15)=ax0;  { FFT block exponent }
	ax0=32;	dm(status_buf+16)=ax0;	{ PF - Programmable Flags }
	ax0=2;	dm(status_buf+17)=ax0;  { Remain 2; used as valid data check }
	ax0=18;	dm(status_buf+18)=ax0;  { Low 8 bits of PM (see +7) }
	{ The last byte must never be ESC_CHAR }
	ax0=0X0A; dm(status_buf+19)=ax0;  { Was 1, don't use for data (rev 2A)}

	ax0=^esc_a;		dm(tx_buf_ptr+0)=ax0;
	ax0=^UART_tx_data_buf0;	dm(tx_buf_ptr+1)=ax0;
	ax0=^status_buf;	dm(tx_buf_ptr+2)=ax0;
	ax0=2;			dm(tx_buf_len+0)=ax0;
	ax0=513;		dm(tx_buf_len+1)=ax0;  { 512 data+check sum }
	ax0=20;			dm(tx_buf_len+2)=ax0;
	ax0=0;			dm(tx_buf_word_len+0)=ax0;
	ax0=0;			dm(tx_buf_word_len+1)=ax0;
	ax0=0;			dm(tx_buf_word_len+2)=ax0;
	{Set first buffer to be transmitted.  Note that this need not be
	 the first of the standard sequence, but could be a power up string.
	 Not this time, however.  }
	ax0=^esc_a;
	dm(UART_tx_ptr)=ax0;
	ax0=0;
	dm(UART_tx_word_len)=ax0;	{ Send bytes }
	dm(UART_tx_byte_num) = ax0;	{ Lower 8 bits }
	dm(UART_tx_cur_data)=ax0;
	dm(UART_tx_esc) = ax0;		{ Not esc in data, yet }
	ax0=2;
	dm(UART_tx_ch_count) = ax0;	{ Number of char left; 0=empty }
	ax0=10;
	dm(UART_tx_cur_bit) = ax0;	{ Bit being sent, 0=finished }
	ax0 = 1000;
	dm(UART_tx_resync_count) = ax0;
	ax0=DATA_BLOCKS-1;
	dm(UART_tx_data_block)=ax0;
	{Init xmit status that is sent in place of log data during transmit }
	ax0=0x30;
	dm(xmit_status_buf+0)=ax0;
	dm(xmit_status_buf+1)=ax0;
	dm(xmit_status_buf+2)=ax0;
	dm(xmit_status_buf+3)=ax0;
	dm(xmit_status_buf+4)=ax0;
	dm(xmit_status_buf+5)=ax0;
	dm(xmit_status_buf+6)=ax0;
	dm(xmit_status_buf+7)=ax0;
	dm(xmit_status_buf+8)=ax0;
	dm(xmit_status_buf+9)=ax0;

{****************** Initialize UART Rcvr  ***********************************}
	ax0=0x20;				{ Initialize cmd_buf }
	dm(UART_rx_cmd_count)=ax0;
	dm(cmd_buf+0)=ax0;
	dm(cmd_buf+1)=ax0;
	dm(cmd_buf+2)=ax0;
	dm(cmd_buf+3)=ax0;
	dm(cmd_buf+4)=ax0;
	ax0=^cmd_buf;
	dm(UART_rx_cmd_buf_ptr)=ax0;	
	ax0=0;
	dm(UART_rx_cur_bit)=ax0;		{ Presently nothing active }
	dm(UART_rx_cur_byte)=ax0;
	ax0=5;
	dm(UART_rx_ctr)=ax0;			{ Bit periods 1 to 5 }
{****************************************************************************}
        ax0 = 2; 
        dm(integrate_count) = ax0;
        ax0 = 19114;			{ 14 KHz }
	dm(if_dphase) = ax0;
	ax0=0;
	dm(if_phase)=ax0;

        i0=^widata;		{ <<<<<<<<<<<<<<<<<<<<<<<<<<<< }
        dm(wfir1i_i0_sav)=i0;		{ Save i0 for interrupt; protects fft }


        i0=^idata;  m0=1; l0=TAPS1;	{ Output of I software mixer }
        dm(fir1i_i0_sav)=i0;		{ Save i0 for interrupt; protects fft }
        cntr=TAPS1;
        do zero1i until ce;
zero1i:      dm(i0,m0)=0;                { clear 'data' fir delay line buffer }
        i0=^qdata;  m0=1; l0=TAPS1;	{ Output of Q software mixer }
        dm(fir1q_i0_sav)=i0;		{ Save i0 for interrupt; protects fft }
        cntr=TAPS1;
        do zero1q until ce;
zero1q:      dm(i0,m0)=0;		{ clear 'data' fir delay line buffer }

        i0=^audio_data; m0=1; l0=TAPS2;	{ Audio ready for output }
        { Save DAG settings for use during interrupt; protects fft }
        dm(fir2_i0_sav)=i0;
        cntr=TAPS2;
        do zero2 until ce;
zero2:       dm(i0,m0)=0;		{ clear fir2 delay line buffer }

        i0=^audio_fir_data; m0=1; l0=TAPS3; { Audio from i-q, to be filtered }
        { Save DAG settings for use during interrupt; protects fft }
        dm(fir3_i0_sav)=i0;
        cntr=TAPS3;
        do zero3 until ce;
zero3:       dm(i0,m0)=0;		{ clear fir3 delay line buffer }

        i0=^xmt_audio_data; m0=1; l0=TAPS4; { Audio for transmitter }
        { Save DAG settings for use during interrupt; protects fft }
        dm(fir4_i0_sav)=i0;
        cntr=TAPS4;
        do zero4 until ce;
zero4:       dm(i0,m0)=0;		{ clear fir4 delay line buffer }

        i0=^xmt_cw_data; m0=1; l0=TAPS5; { CW Spectrum shaping }
        dm(fir5_i0_sav)=i0;
        cntr=TAPS5;
        do zero5 until ce;
zero5:       dm(i0,m0)=0;		{ clear fir5 delay line buffer }

        i0=^rcv_fm; m0=1; l0=TAPS6; 	{ FM Rcvr HP Filt }
        dm(fir6_i0_sav)=i0;
        cntr=TAPS6;
        do zero6 until ce;
zero6:       dm(i0,m0)=0;		{ clear fir6 delay line buffer }

        i0=^rcv1_fm; m0=1; l0=TAPS7; 	{ FM Rcvr LP Filt }
        dm(fir7_1_i0_sav)=i0;
        cntr=TAPS7;
        do zero7_1 until ce;
zero7_1:       dm(i0,m0)=0;		{ clear delay line buffer }

        i0=^rcv2_fm; m0=1; l0=TAPS7; 	{ FM Rcvr LP Filt }
        dm(fir7_2_i0_sav)=i0;
        cntr=TAPS7;
        do zero7_2 until ce;
zero7_2:       dm(i0,m0)=0;		{ clear delay line buffer }

        i0=^lp1200; m0=1; l0=TAPS8; 	{ 1200 LPF for Nyquist }
        dm(fir1200_i0_sav)=i0;
        cntr=TAPS8;
        do zero1200 until ce;
zero1200:    dm(i0,m0)=0;		{ clear delay line buffer }

        i0=^lp2400; m0=1; l0=TAPS8; 	{ 2400 LPF for Nyquist }
        dm(fir2400_i0_sav)=i0;
        cntr=TAPS8;
        do zero2400 until ce;
zero2400:    dm(i0,m0)=0;		{ clear delay line buffer }

        i0=^UART_tx_data_buf0; m0=1; l0=0;
        ax0=52; cntr=512;		{ '4' }
        do bl1s until ce;
bl1s:       dm(i0,m0)=ax0;

        i0=^UART_tx_data_buf1; m0=1; l0=0;
        ax0=53; cntr=512;		{ '5' }
        do bl2s until ce;
bl2s:       dm(i0,m0)=ax0;

	ax0=0; ay0=1;
	i0=^out_log;  m0=1; l0=0;
	cntr=512;
	do init_log until ce;
	    ar=ax0+ay0;
	    ax0=ar;
	    ay1=63; ax1=ar; ar=ax1 and ay1;
	    ay1=47; ax1=ar; ar=ax1+ay1;
init_log:   dm(i0, m0)=ar;

	i0=^h1delay; m0=1; l0=%h1delay;
	dm(h1delay_i0_sav)=i0;
	cntr=%h1delay;
	do zeroh1dly until ce;
zeroh1dly:  dm(i0,m0)=0;

	i0=^h1data; m0=1; l0=%h1data;
	dm(h1data_i0_sav)=i0;
	cntr=%h1data;
	do zeroh1dat until ce;
zeroh1dat:  dm(i0,m0)=0;

	i0=^h2delay; m0=1; l0=%h2delay;
	dm(h2delay_i0_sav)=i0;
	cntr=%h2delay;
	do zeroh2dly until ce;
zeroh2dly:  dm(i0,m0)=0;

	i0=^h2data; m0=1; l0=%h2data;
	dm(h2data_i0_sav)=i0;
	cntr=%h2data;
	do zeroh2dat until ce;
zeroh2dat:  dm(i0,m0)=0;

	i0=^h3delay; m0=1; l0=%h3delay;
	dm(h3delay_i0_sav)=i0;
	cntr=%h3delay;
	do zeroh3dly until ce;
zeroh3dly:  dm(i0,m0)=0;

	i0=^h3data; m0=1; l0=%h3data;
	dm(h3data_i0_sav)=i0;
	cntr=%h3data;
	do zeroh3dat until ce;
zeroh3dat:  dm(i0,m0)=0;

	rts;

#if CODEC_1847
{************************  NEXT_CMD  ****************************************
 * Transmit Interrupt used for Codec initialization
 * DAG 5 registers used here, but are available after this routine is run.
 * Adapted from EZ-KIT Lite routines.
 ****************************************************************************}
next_cmd:
        ena sec_reg;
        ax0 = dm (i5, m5);          { fetch next control word and }
        dm (tx_buf) = ax0;          { place in transmit slot 0    }
        ax0 = i5;
        ay0 = ^init_cmds;
        ar = ax0 - ay0;
        if gt rti;                  { rti if more control words still waiting }
        ax0 = 0x8000;               { else set done flag and }
        dm (tx_buf) = ax0;          { remove MCE if done initialization }
        ax0 = 0;
        dm (stat_flag) = ax0;       { reset status flag }
        dis sec_reg;
        rti;
#endif

#if CODEC_AIC23
{----------------------------------------------------------------------------}
{      Subroutine used for Codec initialization, Rev C Module	   	     }
{									     }
{ This subroutine outputs the 16-bit value in ax0 to the		     }
{ TLV320AIC23 codec control port, set up as a SPI port.			     }
{									     }
{ It works with the ADSP-218xN running at maximum speed (80 MIPs).	     }
{									     }
{ The data is shifted out as a 16-bit value, MSB first, per		     }
{ the AIC data sheet.							     }
{									     }
{ Registers altered:							     }
{	ar, sr, ax1, ay1, af, status flag				     }
{									     }
{ Written:  10 Feb 2003 L Johnson (Rev B)				     }
{ Modified: 21 Feb 2003 L Johnson (Rev C)				     }
{									     }
{----------------------------------------------------------------------------}
CODEC_CMD:
	ax1=16;			{ bit count }
	ay1=SPICS+SPIDAT;	{ mask to set SPICLK low }
	sr0=ax0; 		{ data to shift out }
	ar=SPICS;		{ SPICS high, SPICLK low, SPIDAT low }
	io(CODEC)=ar;		{ initial conditions }
	ar=ar and SPIDAT;	{ SPICS low }
	nop; nop;		{ delay for AIC }
	io(CODEC)=ar;		{ now ready to send cmd }

CODEC1:
	ar=0;			{ set data bit = 0 }
	af=sr0 AND 0x8000;	{ if MSB is set.. }
	if ne ar=ar or SPIDAT;	{ ..data bit = 1 }
	io(CODEC)=ar;		{ output data bit }
	nop; nop;		{ ..delay for AIC }
	ar=ar or SPICLK;	{ ..pulse SPLICLK high }
	io(CODEC)=ar;
	nop; nop;		{ ..delay for AIC }
	ar=ar and SPIDAT;	{ ..bring SPICLK low again }
	io(CODEC)=ar;
	ar=ax1-1;		{ decrement bit counter }
	if eq jump CODEC2;      { exit if last bit sent, else..}
	ax1=ar;			{ ..save bit counter, }
	sr=lshift sr0 by 1(LO);	{ ..promote next bit to MSB, }
	jump CODEC1;		{ ..and loop until done }
CODEC2:				{ clean up and exit }
	ar=ar or SPICS;		{ SPICS high to latch cmd, }
	nop; nop;		{ ..delay for AIC }
	io(CODEC)=ar;		{ ..output CS bit }
	rts;			{ ..and return to caller }
#endif
.endmod;
